<?php
/**
 * Admin bootstrap: config + session
 */

declare(strict_types=1);

define('LICENSE_SERVER', true);

$configPath = dirname(__DIR__) . '/config.php';
if (!is_readable($configPath)) {
    die('Configuration not found.');
}
$config = require $configPath;

$sessionName = $config['session']['name'] ?? 'SSA_LICENSE_ADMIN';
if (session_status() === PHP_SESSION_NONE) {
    session_name($sessionName);
    if (isset($config['session']['lifetime'])) {
        ini_set('session.gc_maxlifetime', (string) $config['session']['lifetime']);
    }
    session_start();
}

$dsn = sprintf(
    'mysql:host=%s;dbname=%s;charset=%s',
    $config['db']['host'],
    $config['db']['name'],
    $config['db']['charset']
);
try {
    $pdo = new PDO($dsn, $config['db']['user'], $config['db']['password'], [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
} catch (PDOException $e) {
    die('Database connection failed.');
}

function admin_logged_in(): bool {
    return !empty($_SESSION['admin_user_id']);
}

function require_admin(): void {
    if (!admin_logged_in()) {
        header('Location: login.php');
        exit;
    }
}

function get_client_ip(): string {
    $keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
    foreach ($keys as $key) {
        if (!empty($_SERVER[$key])) {
            $ip = trim(explode(',', (string) $_SERVER[$key])[0]);
            if (filter_var($ip, FILTER_VALIDATE_IP)) return $ip;
        }
    }
    return '0.0.0.0';
}
